/***************************************************************************//**
 * @file
 * @brief Certificate Based Authentication and Pairing implementation
 *******************************************************************************
 * # License
 * <b>Copyright 2022 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * SPDX-License-Identifier: Zlib
 *
 * The licensor of this software is Silicon Laboratories Inc.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 *
 ******************************************************************************/

#include <stdbool.h>
#include "em_common.h"
#include "em_system.h"

#include "app_assert.h"
#include "gatt_db.h"
#include "sl_bluetooth.h"
#include "app_timer.h"
#include "sl_bt_cbap_config.h"
#include "sl_bt_cbap_root_cert.h"
#include "sl_bt_cbap.h"
#include "sl_bt_cbap_lib.h"

#ifdef SL_COMPONENT_CATALOG_PRESENT
#include "sl_component_catalog.h"
#endif // SL_COMPONENT_CATALOG_PRESENT
#ifdef SL_CATALOG_APP_LOG_PRESENT
#include "app_log.h"
#endif // SL_CATALOG_APP_LOG_PRESENT

// -----------------------------------------------------------------------------
// Defines

#if defined(SL_CATALOG_APP_LOG_PRESENT) && SL_BT_CBAP_LOG
#define sl_bt_cbap_log_debug(...)           app_log_debug(__VA_ARGS__)
#define sl_bt_cbap_log_info(...)            app_log_info(__VA_ARGS__)
#define sl_bt_cbap_log_error(...)           app_log_error(__VA_ARGS__)
#define sl_bt_cbap_log_hexdump(p_data, len) app_log_hexdump_debug(p_data, len)
#else
#define sl_bt_cbap_log_debug(...)
#define sl_bt_cbap_log_info(...)
#define sl_bt_cbap_log_error(...)
#define sl_bt_cbap_log_hexdump(p_data, len)
#endif

#define IS_PERIPHERAL_IN_PROGRESS (cbap_peripheral_state > 0 && cbap_peripheral_state < SL_BT_CBAP_PERIPHERAL_STATE_NUM - 1)
#define IS_CENTRAL_IN_PROGRESS    (cbap_central_state > 0 && cbap_central_state < SL_BT_CBAP_CENTRAL_STATE_NUM - 1)

#define UUID_16_LEN                   2
#define UUID_128_LEN                  16
#define HANDLE_NOT_INITIALIZED        0

#define GAP_INCOMPLETE_16B_UUID       0x02 // Incomplete List of 16-bit Service Class UUIDs
#define GAP_COMPLETE_16B_UUID         0x03 // Complete List of 16-bit Service Class UUIDs
#define GAP_INCOMPLETE_128B_UUID      0x06 // Incomplete List of 128-bit Service Class UUIDs
#define GAP_COMPLETE_128B_UUID        0x07 // Complete List of 128-bit Service Class UUIDs

#define CHAIN_LINK_DATA_LEN           192  // Length of an NVM3 chunk
#define CHAIN_LINK_DATA_NUM           4    // Number of how many chunks needed for a certificate

#define OOB_SIGNATURE_LEN             64
#define CERT_IND_CHUNK_LEN            100
#define EC_PUB_KEY_LEN                65
#define PUB_KEY_OFFSET                26

#define OOB_RANDOM_LEN                (sizeof(aes_key_128))
#define OOB_DATA_LEN                  (2 * OOB_RANDOM_LEN)
#define SIGNED_OOB_DATA_LEN           (OOB_DATA_LEN + OOB_SIGNATURE_LEN)

#define TIMEOUT                       5000 // ms
#define NO_CALLBACK_DATA              (void *)NULL // Callback has no parameters

// -----------------------------------------------------------------------------
// Type definitions.

typedef struct characteristic_128_ref_s {
  uint16_t handle;
  uint8_t uuid[UUID_128_LEN];
} characteristic_128_ref_t;

typedef enum {
  CHAR_CENTRAL_CERT,
  CHAR_PERIPHERAL_CERT,
  CHAR_CENTRAL_OOB,
  CHAR_PERIPHERAL_OOB,
  CHAR_NUM
} characteristics_t;

// -----------------------------------------------------------------------------
// Module variables.

// Device role
static sl_bt_cbap_role_t role;

// Handle of the active connection.
static uint8_t connection = SL_BT_INVALID_CONNECTION_HANDLE;

// Root certificate in PEM format.
const char *root_certificate_pem = SL_BT_CBAP_ROOT_CERT;

// Reference to the CBAP service.
static uint32_t cbap_service_handle = HANDLE_NOT_INITIALIZED;
static const uint8_t cbap_service_uuid[] = {
  0x10, 0x56, 0x28, 0xd0, 0x40, 0xdd, 0x8e, 0x91,
  0x4d, 0x41, 0x47, 0x81, 0xc6, 0x8c, 0x81, 0xd8
};

// Reference to the CBAP characteristics.
static characteristic_128_ref_t cbap_characteristics[] = {
  {   // CHAR_CENTRAL_CERT
    .handle = HANDLE_NOT_INITIALIZED,
    .uuid = {
      0xca, 0x25, 0xf3, 0xa2, 0xf6, 0xda, 0xbd, 0xb6,
      0x69, 0x4a, 0xaa, 0x08, 0xf9, 0xd0, 0x5f, 0x31
    }
  },
  {   // CHAR_PERIPHERAL_CERT
    .handle = HANDLE_NOT_INITIALIZED,
    .uuid = {
      0x66, 0x50, 0xfd, 0x84, 0x4d, 0xad, 0xa2, 0x99,
      0xc9, 0x4f, 0xf5, 0x16, 0x9e, 0xda, 0xf6, 0x0c
    }
  },
  {   // CHAR_CENTRAL_OOB
    .handle = HANDLE_NOT_INITIALIZED,
    .uuid = {
      0x2c, 0x19, 0xf1, 0xeb, 0x85, 0xcd, 0xb6, 0x8a,
      0x2c, 0x4e, 0x7d, 0x89, 0x51, 0x57, 0xd3, 0xe8
    }
  },
  {   // CHAR_PERIPHERAL_OOB
    .handle = HANDLE_NOT_INITIALIZED,
    .uuid = {
      0xab, 0x69, 0xb2, 0x5e, 0xb8, 0x41, 0xb3, 0x8b,
      0x82, 0x41, 0x51, 0xa2, 0x41, 0xcb, 0x91, 0x69
    }
  }
};

// State of the central device
static sl_bt_cbap_central_state_t cbap_central_state = SL_BT_CBAP_CENTRAL_SCANNING;
// Pointing to the characteristic that shall be discovered next
static characteristics_t char_state = (characteristics_t)0;
// State of the peripheral device
static sl_bt_cbap_peripheral_state_t cbap_peripheral_state = SL_BT_CBAP_PERIPHERAL_IDLE;

// Timer handle
static app_timer_t state_timer;

// Remote certificate which was sent over GATT in DER format
static uint8_t remote_certificate_der[CHAIN_LINK_DATA_LEN * CHAIN_LINK_DATA_NUM] = { 0 };
static uint32_t remote_certificate_der_len = 0;
static bool remote_cert_arrived = false;

// Device certificate in DER format
static uint8_t device_certificate_der[CHAIN_LINK_DATA_LEN * CHAIN_LINK_DATA_NUM] = { 0 };
static uint32_t device_certificate_der_len = 0;
static uint32_t dev_cert_sending_progression = 0;
static bool device_cert_sent = false;

static uint8_t signed_device_oob_data[SIGNED_OOB_DATA_LEN];
static size_t signed_device_oob_len = 0;

// -----------------------------------------------------------------------------
// Private function declarations

// Central device bluetooth event handler.
static void on_event_peripheral(sl_bt_msg_t *evt);
// Peripheral device bluetooth event handler.
static void on_event_central(sl_bt_msg_t *evt);

// Reset CBAP process states, flags and timers.
static void cbap_reset(void);

// Search for a Service UUID in scan report.
static bool find_service_in_advertisement(const uint8_t *scan_data,
                                          uint8_t scan_data_len,
                                          const uint8_t *uuid,
                                          uint8_t uuid_len);

// Start or stop timer for timeout check.
static void set_timeout(bool activate);
// Timer callback.
static void state_timer_cb(app_timer_t *handle, void *data);

// -----------------------------------------------------------------------------
// Public function definitions

// Initialize the component. Import and validate the device and root certificate.
void sl_bt_cbap_init(void)
{
  sl_status_t sc;
  sc = sl_bt_cbap_lib_init(root_certificate_pem,
                           device_certificate_der,
                           &device_certificate_der_len);
  app_assert_status(sc);
  sl_bt_cbap_log_info("Device certificate verified." APP_LOG_NL);

  cbap_reset();
}

// Start CBAP procedure.
sl_status_t sl_bt_cbap_start(sl_bt_cbap_role_t cbap_role,
                             uint8_t connection_handle)
{
  sl_status_t sc;
  if (IS_PERIPHERAL_IN_PROGRESS || IS_CENTRAL_IN_PROGRESS) {
    return SL_STATUS_IN_PROGRESS;
  }

  role = cbap_role;
  connection = connection_handle;

  if (role == SL_BT_CBAP_ROLE_CENTRAL) {
    // Discover CBAP service on the peripheral device
    sc = sl_bt_gatt_discover_primary_services_by_uuid(connection,
                                                      sizeof(cbap_service_uuid),
                                                      (const uint8_t *)cbap_service_uuid);
    app_assert_status(sc);
    cbap_central_state = SL_BT_CBAP_CENTRAL_DISCOVER_SERVICES;
    sl_bt_cbap_central_on_event(cbap_central_state);
  }

  set_timeout(true);
  return SL_STATUS_OK;
}

// Bluetooth stack event handler.
void sli_bt_cbap_on_event(sl_bt_msg_t *evt)
{
  switch (role) {
    case SL_BT_CBAP_ROLE_PERIPHERAL:
      on_event_peripheral(evt);
      break;
    case SL_BT_CBAP_ROLE_CENTRAL:
      on_event_central(evt);
      break;
  }
}

// Search for a the CBAP Service UUID in scan report.
bool sl_bt_cbap_find_service_in_advertisement(const uint8_t *scan_data,
                                              uint8_t scan_data_len)
{
  return find_service_in_advertisement(scan_data,
                                       scan_data_len,
                                       cbap_service_uuid,
                                       sizeof(cbap_service_uuid));
}

// -----------------------------------------------------------------------------
// Private function definitions

/*******************************************************************************
 * Peripheral device bluetooth event handler.
 ******************************************************************************/
static void on_event_peripheral(sl_bt_msg_t *evt)
{
  sl_status_t sc;
  // Handle stack events
  switch (SL_BT_MSG_ID(evt->header)) {
    // This event indicates the device has started and the radio is ready.
    // Do not call any stack command before receiving this boot event!
    case sl_bt_evt_system_boot_id:
      // Request OOB data from both device
      sc = sl_bt_sm_configure(SL_BT_SM_CONFIGURATION_OOB_FROM_BOTH_DEVICES_REQUIRED,
                              sl_bt_sm_io_capability_noinputnooutput);
      app_assert_status(sc);
      break;

    //--------------------------------
    // Triggered whenever the connection parameters are changed
    case sl_bt_evt_connection_parameters_id:
      if (evt->data.evt_connection_parameters.connection != connection) {
        break;
      }

      sl_bt_cbap_log_debug("Security mode: %i" APP_LOG_NL,
                           evt->data.evt_connection_parameters.security_mode);
      if (evt->data.evt_connection_parameters.security_mode > sl_bt_connection_mode1_level1
          && cbap_peripheral_state != SL_BT_CBAP_PERIPHERAL_CENTRAL_OOB_OK                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                